/* Copyright (C) 2012 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_cff.c */


#include "fs_object.h"
/*#define CFF_DEBUG*/

#ifdef FS_CFFR        /* conditionally compile entire module */

#include "fs_function.h"
#include "fs_err.h"
#include "fs_fixed.h"
#include "fs_fnt.h"
#include "fs_scratch.h"


typedef struct
{
    FS_LONG val[48];
    FS_BYTE type[48];    /* one of the ARG_TYPE_xxx */
} ARGS;

#define ARG_TYPE_INT 1
#define ARG_TYPE_FIXED 2    /* 16.16 */
#define ARG_TYPE_FRACT 3    /* 2.30 */

/* the following are encoded by ranges */
static FS_CONST FS_USHORT ISOAdobeSID[] =
{
    1,  228,
    0,    0
};

static FS_CONST FS_USHORT ExpertSID[] =
{
    1,      1,
    229,    238,
    13,     15,
    99,     99,
    239,    248,
    27,     28,
    249,    266,
    109,    110,
    267,    318,
    158,    158,
    155,    155,
    163,    163,
    319,    326,
    150,    150,
    164,    164,
    169,    169,
    327,    378,
    0,      0
};

static FS_CONST FS_USHORT ExpertSubsetSID[] =
{
    1,      1,
    231,    232,
    235,    238,
    13,     15,
    99,     99,
    239,    248,
    27,     28,
    249,    251,
    253,    266,
    109,    110,
    267,    270,
    272,    272,
    300,    302,
    305,    305,
    314,    315,
    158,    158,
    155,    155,
    163,    163,
    320,    326,
    150,    150,
    164,    164,
    169,    169,
    327,    346,
    0,      0
};

/* prototypes */
static int getINDEXsizefromoffset(_DS_ TTF *ttf, int off, FS_USHORT *count);
static void *getINDEXptr(FS_BYTE *ptr, int index, FS_ULONG *size);
static void initTopDict(CFFTOPDICT *topdict);
static void loadTopDict(TTF *ttf, FS_LONG size, FS_BYTE *ptr, CFFTOPDICT *topdict);
static void loadPrivateDict(FS_BYTE *ptr, FS_LONG size, PRIVATEDICT *privatedict);
static FS_SHORT nextopcode( FS_BYTE** pdata, FS_LONG* numargs, ARGS *args, FS_BYTE *ep);
static FS_BYTE *get_real(FS_BYTE *p, ARGS *args, int argcount);

#ifdef FS_RENDER      /* otherwise nothing to do */

/********************************************************************************/
typedef struct
{
    int x0, y0;
    int x1, y1;
    int x2, y2;
    int x3, y3;
    int x4, y4;
    int x5, y5;
    int x6, y6;
    int fd;
} FLEXARGS;

/********************************************************************************/
FS_FIXED fix_arg(ARGS *args, int n)
{
    FS_LONG v = args->val[n];
    FS_BYTE t = args->type[n];
    if (t == ARG_TYPE_INT)
        v <<= 16;
    else if (t == ARG_TYPE_FRACT)
        v >>= 14;
    return v;
}

/* externs */
extern FS_VOID scale_char(fsg_SplineKey *key, fnt_ElementType *elementPtr);
extern FS_VOID fsg_CopyElementBackwards(fnt_ElementType *elementPtr);
extern FS_VOID apply_transMatrix(fsg_SplineKey *key, F26DOT6 *x, F26DOT6 *y, int np, transMatrix *trans);

/* prototypes */
static void *getINDEXptrfromoffset(_DS_ TTF *ttf, int off, int index, int *size);
static void *getSubrINDEXptr(_DS_ FS_BYTE *ptr, int index, FS_LONG *size);
static int close_contour(_DS_ CFF_DATA *cffoutl);
static int moveto(_DS_ int x, int y, CFF_DATA *cffoutl);
static int lineto(_DS_ int x, int y, CFF_DATA *cffoutl);
static int curveto(_DS_ int x1, int y1, int x2, int y2, int x3, int y3, CFF_DATA *cffoutl);
static int flexproc(_DS_ FLEXARGS *fa, CFF_DATA *cffoutl);
static FS_BYTE *do_hintmask(FS_BYTE *ptr, CFF_DATA *cffoutl);
static FS_BYTE *do_cntrmask(FS_BYTE *ptr, CFF_DATA *cffoutl);
static FS_SHORT nextcsopcode( FS_BYTE** pdata, int* numargs, int *args, FS_BYTE *ep);
static FS_ULONG generateoutline(_DS_ TTF *ttf, FS_BYTE *ptr, FS_LONG size, CFF_DATA *cffoutl);
static FS_ULONG processseac(_DS_ TTF *ttf, FS_FIXED adx, FS_FIXED ady, int bchar, int achar, CFF_DATA *cffoutl);

#define MOVETO(x,y) \
    { err=moveto(_PS_ x,y,cffoutl); if (err) return err; }
#define LINETO(x,y) \
    { err=lineto(_PS_ x,y,cffoutl); if (err) return err; }
#define CURVETO(x1,y1,x2,y2,x3,y3) \
    { err=curveto(_PS_ x1,y1,x2,y2,x3,y3,cffoutl); if (err) return err; }
#define FLEXPROC(fa)\
    { err=flexproc(_PS_ fa,cffoutl); if (err) return err; }

#define NEXTARG() ((args[i++]+32) >> 10)
#define NEXTARGNONCOORD() (args[i++] >> 16)
#define HSTEM 1
#define VSTEM 2

#define POPCFFARG() args[--cffoutl->numargs]
#define PUSHCFFARG(x) args[cffoutl->numargs++] = x;

/* Adobe standard encoding vector */
#define ADOBESTDENCODINGOFFSET 32  /* first 32 entries are zero so not included */
#define ADOBESTDENCODINGLENGTH 224 /* multiple of 8 bytes       */
static FS_CONST FS_BYTE adobestdvector[ADOBESTDENCODINGLENGTH] =
{
    1,    2,    3,    4,    5,    6,    7,    8,
    9,   10,   11,   12,   13,   14,   15,   16,
    17,   18,   19,   20,   21,   22,   23,   24,
    25,   26,   27,   28,   29,   30,   31,   32,
    33,   34,   35,   36,   37,   38,   39,   40,
    41,   42,   43,   44,   45,   46,   47,   48,
    49,   50,   51,   52,   53,   54,   55,   56,
    57,   58,   59,   60,   61,   62,   63,   64,
    65,   66,   67,   68,   69,   70,   71,   72,
    73,   74,   75,   76,   77,   78,   79,   80,
    81,   82,   83,   84,   85,   86,   87,   88,
    89,   90,   91,   92,   93,   94,   95,    0,
    0,    0,    0,    0,    0,    0,    0,    0,
    0,    0,    0,    0,    0,    0,    0,    0,
    0,    0,    0,    0,    0,    0,    0,    0,
    0,    0,    0,    0,    0,    0,    0,    0,
    0,   96,   97,   98,   99,  100,  101,  102,
    103,  104,  105,  106,  107,  108,  109,  110,
    0,  111,  112,  113,  114,    0,  115,  116,
    117,  118,  119,  120,  121,  122,    0,  123,
    0,  124,  125,  126,  127,  128,  129,  130,
    131,    0,  132,  133,    0,  134,  135,  136,
    137,    0,    0,    0,    0,    0,    0,    0,
    0,    0,    0,    0,    0,    0,    0,    0,
    0,  138,    0,  139,    0,    0,    0,    0,
    140,  141,  142,  143,    0,    0,    0,    0,
    0,  144,    0,    0,    0,  145,    0,    0,
    146,  147,  148,  149,    0,    0,    0,    0
};

#ifdef CFF_DEBUG
int indent = 0;

FS_VOID dump_element(char *s, fnt_ElementType *elementPtr)
{
    int p, c, nc = elementPtr->nc;
    FS_BYTE *o = elementPtr->onCurve;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    FS_LONG *oox = elementPtr->oox;
    FS_LONG *ooy = elementPtr->ooy;

    FS_PRINTF(("dump_element -- %s\n", s));
    FS_PRINTF(("nc %d\n", elementPtr->nc));
    if (nc)
    {
        FS_PRINTF(("sp[] = "));
        for (c = 0; c < nc; c++)
            FS_PRINTF(("%d ", sp[c]));
        FS_PRINTF(("\n"));

        FS_PRINTF(("ep[] = "));
        for (c = 0; c < nc; c++)
            FS_PRINTF(("%d ", ep[c]));
        FS_PRINTF(("\n"));

        for (c = 0; c < nc; c++)
        {
            for (p = sp[c]; p <= ep[c]; p++)
                FS_PRINTF(("%3d %d %7.3f %7.3f %5ld %5ld\n", p, o[p], x[p] / 64.0, y[p] / 64.0, oox[p], ooy[p]));
            p = sp[c];
            FS_PRINTF(("%3d %d %7.3f %7.3f %5ld %5ld\n", p, o[p], x[p] / 64.0, y[p] / 64.0, oox[p], ooy[p]));
            FS_PRINTF(("\n"));
        }
    }
    FS_FFLUSH(stdout);
}
#endif /* CFF_DEBUG */

/********************************************************************************/
static void *getINDEXptrfromoffset(_DS_ TTF *ttf, int off, int index, int *size)
{
    FS_USHORT count;
    FS_BYTE entrysize;
    int i, offset, nextoffset;
    FS_BYTE tmp[8];
    void *r;

    *size = 0; /* in case read fails */

    ttf_read_buf(_PS_ ttf, off, 3, tmp);
    count = (tmp[0] << 8) | tmp[1];
    entrysize = tmp[2];

    ttf_read_buf(_PS_ ttf, off + 3 + index * entrysize, 8, tmp);

    if (count < index)
        return NULL;

    for (offset = 0, nextoffset = 0, i = 0; i < entrysize; i++)
    {
        offset = (offset << 8) + tmp[i];
        nextoffset = (nextoffset << 8) + tmp[i + entrysize];
    }

    *size = nextoffset - offset;

    r = FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr, *size);
    if (r)
        ttf_read_buf(_PS_ ttf, off + 3 + (count + 1)*entrysize - 1 + offset, *size, r);
    return r;
}

/********************************************************************************/
static void *getSubrINDEXptr(_DS_ FS_BYTE *ptr, int index, FS_LONG *size)
{
    FS_USHORT count;
    int entrysize;
    int i, offset, nextoffset;
    FS_BYTE *p = ptr;

    count = (p[0] << 8) | p[1];
    entrysize = p[2];
    p += 3;

    if (count < 1240)
        index = index + 107;
    else if (count < 33900)
        index = index + 1131;
    else
        index = index + 32768;

    if (count < index)
    {
        STATE.error = ERR_CFF_BAD_SUBRSPTR;
        return NULL;
    }

    for (offset = 0, nextoffset = 0, i = 0; i < entrysize; i++)
    {
        offset = (offset << 8) + *(p + index * entrysize + i);
        nextoffset = (nextoffset << 8) + *(p + (index + 1) * entrysize + i);
    }

    *size = nextoffset - offset;
    return (p + (count + 1) * entrysize - 1 + offset);
}

/********************************************************************************/
static void *getSubrINDEXptrfromoffset(_DS_ TTF *ttf, int off, int index, FS_LONG *size)
{
    FS_USHORT count;
    int entrysize;
    int i, offset, nextoffset;
    FS_BYTE *p, tmp[8];

    *size = 0; /* in case read fails */

    ttf_read_buf(_PS_ ttf, off, 3, tmp);
    if (STATE.error)
        return NULL;

    count = (tmp[0] << 8) | tmp[1];
    entrysize = tmp[2];

    if (count < 1240)
        index = index + 107;
    else if (count < 33900)
        index = index + 1131;
    else
        index = index + 32768;

    if (count < index)
    {
        STATE.error = ERR_CFF_BAD_SUBRSPTR;
        return NULL;
    }

    ttf_read_buf(_PS_ ttf, off + 3 + index * entrysize, 8, tmp);
    if (STATE.error)
        return NULL;

    for (offset = 0, nextoffset = 0, i = 0; i < entrysize; i++)
    {
        offset = (offset << 8) + tmp[i];
        nextoffset = (nextoffset << 8) + tmp[i + entrysize];
    }

    *size = nextoffset - offset;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "getSubrINDEXptrfromoffset";
#endif
    /* additional offset amount is (count+1)*entrysize - 1 + 3 + offset */
    p = ttf_read(_PS_ ttf, off + (count + 1) * entrysize + 2 + offset, *size);

    return (void *)p;
}

int getFDindex(_DS_ TTF *ttf, CFF_TAB *cff, int index)
{
    char format;
    FS_BYTE idx;

    if (index >= cff->topdict.CIDcount)
        return 0;

    ttf_read_buf(_PS_ ttf, cff->topdict.FDselectoffset, 1, (FS_BYTE *)&format);

    if (format == 0)
    {
        int offset = cff->topdict.FDselectoffset + 1 + index;
        ttf_read_buf(_PS_ ttf, offset, 1, (FS_BYTE *)&idx);
        return (int)idx;
    }
    else if (format == 3)
    {
        FS_SHORT nRanges;
        int offset = cff->topdict.FDselectoffset + 1;
        FS_SHORT first;
        int i;
        ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&nRanges);
        nRanges = SWAPW(nRanges);
        offset += 2;
        idx = 0;
        for (i = 0; i < nRanges; i++)
        {
            ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&first);
            first = SWAPW(first);
            if (index < first)
                return (int)idx;
            offset += 2;
            ttf_read_buf(_PS_ ttf, offset, 1, (FS_BYTE *)&idx);
            offset++;
        }

        /* return index for last range */
        return (int)idx;
    }

    return 0;
}

static int resetCIDdicts(_DS_ TTF *ttf, CFF_TAB *cff, int index)
{
    int FDindex;
    char *topdictptr;
    int size;

    FDindex = getFDindex(_PS_ ttf, cff, index);

    if (cff->topdict.FDselect == FDindex)
        return 0; /* already selected */

    topdictptr = getINDEXptrfromoffset(_PS_ ttf, cff->topdict.FDarrayoffset, FDindex, &size);
    if (topdictptr == NULL)
        return 0;

    loadTopDict(ttf, size, (FS_BYTE *)topdictptr, &cff->topdict);
    cff->topdict.FDselect = FDindex;

    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, topdictptr);

    /* if there is a private dict, load it */
    if (cff->topdict.privatedictsize)
    {
        FS_BYTE *privatedictptr;
        int offset;

        offset = ttf->cff_offset + cff->topdict.privatedictoffset;
        size = cff->topdict.privatedictsize;
#ifdef FS_MEM_DBG
        STATE.memdbgid = "resetCIDdicts";
#endif
        privatedictptr = ttf_read(_PS_ ttf, offset, size);
        loadPrivateDict(privatedictptr, size, &cff->topdict.privatedict);
        FSS_free(_PS_ privatedictptr);
    }

    return 0;
}

/********************************************************************************/
static FS_LONG grow_element(_DS_ int which)
{
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    fnt_ElementType *ep1, *ep2;
    int old_max_points, old_max_contours;
    int max_points, max_contours;
    FS_BYTE *ptr, *old_ptr;
    FS_ULONG size, old_size;

    FS_USHORT nc, *sp, *ep;
    FS_LONG *x, *y, *ox, *oy, *oox, *ooy;
    FS_BYTE *f, *onCurve;

    max_points = old_max_points = key->maxp->maxPoints;
    max_contours = old_max_contours = key->maxp->maxContours;

    /* get existing element pointers */
    ep1 = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    nc = ep1->nc;
    sp = ep1->sp;
    ep = ep1->ep;
    f = ep1->f;
    onCurve = ep1->onCurve;
    x = ep1->x;
    y = ep1->y;
    ox = ep1->ox;
    oy = ep1->oy;
    oox = ep1->oox;
    ooy = ep1->ooy;

    old_ptr = STATE.server->workspace;
    old_size = STATE.server->workspaceSize;

    /* update max_points and max_contours in key->maxp == ttf->maxp */
    if (which & 2)
        max_contours *= 2;
    if (which & 1)
        max_points *= 2;
    key->maxp->maxContours = (FS_USHORT)max_contours;
    key->max_contours = (FS_SHORT)max_contours;
    key->maxp->maxPoints = (FS_USHORT)max_points;
    key->max_points = (FS_SHORT)max_points;

    /* re-compute workspace size and re-calculate offsets     */
    /* then reallocate the workspace and copy old values over */
    size = fsg_WorkSpaceSetOffsets(key);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "grow_element";
#endif
    ptr = FSS_calloc(_PS_ size);
    if (STATE.error)
    {
        /* back out the failed growth */
        key->maxp->maxContours = (FS_USHORT)old_max_contours;
        key->maxp->maxPoints = (FS_USHORT)old_max_points;
        key->max_contours = (FS_SHORT)old_max_contours;
        key->max_points = (FS_SHORT)old_max_points;
        STATE.server->workspace = old_ptr;
        STATE.server->workspaceSize = old_size;
        fsg_WorkSpaceSetOffsets(key);
        fsg_SetUpElement(_PS_ key, GLYPHELEMENT );

        return STATE.error;
    }

    /* get new element */
    STATE.server->workspaceSize = size;
    STATE.server->workspace = ptr;
    fsg_SetUpElement(_PS_ key, GLYPHELEMENT );

    {
        fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
        globalGS->stackBase =  (F26DOT6 *)(STATE.server->workspace + key->elementInfoRec.stackBaseOffset);
    }

    ep2 = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);

    /* copy old to new */
    ep2->nc = nc;

    SYS_MEMCPY(ep2->sp, sp, old_max_contours * sizeof(FS_SHORT));
    SYS_MEMCPY(ep2->ep, ep, old_max_contours * sizeof(FS_SHORT));
    SYS_MEMCPY(ep2->oox, oox, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->ooy, ooy, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->ox, ox, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->oy, oy, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->x, x, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->y, y, old_max_points * sizeof(FS_LONG));
    SYS_MEMCPY(ep2->f, f, old_max_points);
    SYS_MEMCPY(ep2->onCurve, onCurve, old_max_points);

    FSS_free(_PS_ old_ptr);

    return STATE.error = SUCCESS;
}

/********************************************************************************/
static int close_contour(_DS_ CFF_DATA *cffoutl)
{
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    int max_points = key->max_points;
    int nc = elementPtr->nc;
    int np = cffoutl->numpoints;

    if (nc == 0 || np == 0) return STATE.error = SUCCESS;

    /* ? close the current contour */
    if (elementPtr->oox[np - 1] != cffoutl->loopstartx || elementPtr->ooy[np - 1] != cffoutl->loopstarty)
    {
        if ((np + 1) >= max_points)
        {
            grow_element(_PS_ 1);
            if (STATE.error)
                return STATE.error;
            elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
        }

        elementPtr->oox[np] = cffoutl->loopstartx;
        elementPtr->ooy[np] = cffoutl->loopstarty;
        elementPtr->onCurve[np] = 1;
        elementPtr->ep[nc - 1] = (FS_USHORT)np;
        cffoutl->numpoints++;
    }

    return STATE.error = SUCCESS;
}

/********************************************************************************/
static int moveto(_DS_ int x, int y, CFF_DATA *cffoutl)
{
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    int np, nc = elementPtr->nc;
    int xx, yy;

    close_contour(_PS_ cffoutl);
    if (STATE.error)
        return STATE.error;

    /* now get np */
    np = cffoutl->numpoints;

    if ((np + 1) >= key->max_points)
    {
        grow_element(_PS_ 1);
        if (STATE.error)
            return STATE.error;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    }

    if ((nc + 1) >= key->max_contours)
    {
        grow_element(_PS_ 2);
        if (STATE.error)
            return STATE.error;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    }

    /* start a new contour */
    elementPtr->sp[nc] = (FS_USHORT)np;
    elementPtr->nc++;

    /* modify the font coordinates by the fontmatrix[]  */
    /* if not .001 0 0 .001 0 0 ... this is unnecessary */
    {
        TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
        CFF_TAB *cff = ttf->cff;
        FRACT *fontmatrix = cff->topdict.fontmatrix;
        xx = FracMul(x, fontmatrix[0]) + FracMul(y, fontmatrix[2]);
        yy = FracMul(x, fontmatrix[1]) + FracMul(y, fontmatrix[3]);
    }

    /* add the moveto point to the new contour */
    cffoutl->loopstartx = elementPtr->oox[np] = xx;
    cffoutl->loopstarty = elementPtr->ooy[np] = yy;
    elementPtr->onCurve[np] = 1;
    elementPtr->ep[nc] = (FS_USHORT)np;
    cffoutl->numpoints++;

    return STATE.error = SUCCESS;
}

/********************************************************************************/
static int lineto(_DS_ int x, int y, CFF_DATA *cffoutl)
{
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    int max_points = key->max_points;
    int np = cffoutl->numpoints;
    int nc = elementPtr->nc;
    int xx, yy;

    if ((np + 1) >= max_points)
    {
        grow_element(_PS_ 1);
        if (STATE.error)
            return STATE.error;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    }

    /* modify the font coordinates by the fontmatrix[] */
    {
        TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
        CFF_TAB *cff = ttf->cff;
        FRACT *fontmatrix = cff->topdict.fontmatrix;
        xx = FracMul(x, fontmatrix[0]) + FracMul(y, fontmatrix[2]);
        yy = FracMul(x, fontmatrix[1]) + FracMul(y, fontmatrix[3]);
    }

    /* add the point */
    elementPtr->oox[np] = xx;
    elementPtr->ooy[np] = yy;
    elementPtr->onCurve[np] = 1;
    elementPtr->ep[nc - 1] = (FS_USHORT)np;  /* keep ep[] up to date */
    cffoutl->numpoints++;
    return STATE.error = SUCCESS;
}

/********************************************************************************/
static int curveto(_DS_ int x1, int y1, int x2, int y2, int x3, int y3, CFF_DATA *cffoutl)
{
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    int max_points = key->max_points;
    int np = cffoutl->numpoints;
    int nc = elementPtr->nc;
    int xx1, yy1, xx2, yy2, xx3, yy3;

    if ((np + 3) >= max_points)
    {
        grow_element(_PS_ 1);
        if (STATE.error)
            return STATE.error;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    }

    /* modify the font coordinates by the fontmatrix[] */
    {
        TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
        CFF_TAB *cff = ttf->cff;
        FRACT *fontmatrix = cff->topdict.fontmatrix;
        xx1 = FracMul(x1, fontmatrix[0]) + FracMul(y1, fontmatrix[2]);
        yy1 = FracMul(x1, fontmatrix[1]) + FracMul(y1, fontmatrix[3]);
        xx2 = FracMul(x2, fontmatrix[0]) + FracMul(y2, fontmatrix[2]);
        yy2 = FracMul(x2, fontmatrix[1]) + FracMul(y2, fontmatrix[3]);
        xx3 = FracMul(x3, fontmatrix[0]) + FracMul(y3, fontmatrix[2]);
        yy3 = FracMul(x3, fontmatrix[1]) + FracMul(y3, fontmatrix[3]);
    }

    /* add the points */
    elementPtr->oox[np] = xx1;
    elementPtr->ooy[np] = yy1;
    elementPtr->onCurve[np++] = 0;
    elementPtr->oox[np] = xx2;
    elementPtr->ooy[np] = yy2;
    elementPtr->onCurve[np++] = 0;
    elementPtr->oox[np] = xx3;
    elementPtr->ooy[np] = yy3;
    elementPtr->onCurve[np] = 1;
    elementPtr->ep[nc - 1] = (FS_USHORT)np;  /* keep ep[] up to date */

    cffoutl->numpoints += 3;
    return STATE.error = SUCCESS;
}

/********************************************************************************/
static int flexproc(_DS_ FLEXARGS *fa, CFF_DATA *cffoutl)
{
    FS_FIXED x0, y0, x3, y3, x6, y6, A, B, C, dist;
    FIXED_VECTOR T;
    FS_FIXED scale[4];
    TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
    int upm = ttf->head->unitsPerEm;
    FS_FIXED fd = (fa->fd << 16) / 100;  /* from hundredths of a pixel to FIXED pixels */
    int status = SUCCESS;
    int err;    /* for LINETO and CURVETO macros */

    SYS_MEMCPY(scale, STATE.cur_sfnt->user_scale, 16);
    scale[0] /= upm;
    scale[1] /= upm;
    scale[2] /= upm;
    scale[3] /= upm;

    /* transform the interesting points to pixel space */
    /* not to worry... fixed := fixed * int            */
    x0 = scale[0] * fa->x0 + scale[1] * fa->y0;
    y0 = scale[2] * fa->x0 + scale[3] * fa->y0;
    x3 = scale[0] * fa->x3 + scale[1] * fa->y3;
    y3 = scale[2] * fa->x3 + scale[3] * fa->y3;
    x6 = scale[0] * fa->x1 + scale[1] * fa->y6;
    y6 = scale[2] * fa->x1 + scale[3] * fa->y6;

    /* form the normal equation L(x,y) == Ax + By + C              */
    /* note: L(x,y)  == the (signed) distance of <x,y> to the line */
    T.x = y6 - y0; /* A */
    T.y = x0 - x6; /* B */
    fixed_norm(&T); /* now A*A+B*B=ONE */
    A = T.x;
    B = T.y;
    C = FixMul(-y0, B) + FixMul(-x0, A);

    /* evaluate the normal equation at (x3,y3) */
    dist = FixMul(A, x3) + FixMul(B, y3) + C;
    if ( abs(dist) < fd )
    {
        /* flat enough ... use the line */
        LINETO(fa->x6, fa->y6);
    }
    else
    {
        /* not flat enough ... use the curves */
        CURVETO(fa->x1, fa->y1, fa->x2, fa->y2, fa->x3, fa->y3);
        CURVETO(fa->x4, fa->y4, fa->x5, fa->y5, fa->x6, fa->y6);
    }
    return status;
}

static int adobestd(int ch)
{
    if (ch < 32 || ch > 255)
        return 0; /* not found */

    return adobestdvector[(ch - ADOBESTDENCODINGOFFSET)];
}

static int adobeindex(CFF_TAB *cff, int ch)
{
    int i, sid;
    FS_USHORT *charset = cff->topdict.charset;

    if (charset == 0)
        return ch;

    sid = adobestd(ch);

    /* quick return for standard encoding */
    if (charset[sid] == sid)
        return sid;

    for (i = 0; i < cff->topdict.numglyphs; i++ )
    {
        if (charset[i] == sid)
            return i;
    }
    return 0;
}

static FS_ULONG processseac(_DS_ TTF *ttf, FS_FIXED adx, FS_FIXED ady, int bchar, int achar, CFF_DATA *cffoutl)
{
    FS_BYTE *ptr;
    int size;
    CFF_TAB *cff = ttf->cff;
    int aindex, bindex;
    int savedx;
    int savehasadvwidth;
    int err;

    /* first get base character */
    bindex = adobeindex(cff, bchar);
    ptr = (FS_BYTE *)getINDEXptrfromoffset(_PS_ ttf, ttf->cff_offset + cff->topdict.charstringsoffset, bindex, &size);

    /* this should not happen .. but it does ! */
    /* see DFBankBangHK-W5, index=13932        */
    if (ptr == 0 || size <= 0)
        return SUCCESS;

    if (cff->topdict.registry)
    {
        /* CID font - set glyph specific topdict and private dict */
        resetCIDdicts(_PS_ ttf, cff, bindex);
    }

    err = generateoutline(_PS_ ttf, ptr, size, cffoutl);
    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ptr);
    if (err)
        return err;

    /* save advance width of base char */
    savedx = cffoutl->dx;
    savehasadvwidth = cffoutl->hasadvwidth;
    cffoutl->dx = 0;
    cffoutl->hasadvwidth = 0;

    cffoutl->prevx = adx;
    cffoutl->prevy = ady;
    cffoutl->numhstems = 0;
    cffoutl->numvstems = 0;

    aindex = adobeindex(cff, achar);
    ptr = (FS_BYTE *)getINDEXptrfromoffset(_PS_ ttf, ttf->cff_offset + cff->topdict.charstringsoffset, aindex, &size);

    if (ptr == 0 || size <= 0)
        return SUCCESS;

    if (cff->topdict.registry)
    {
        /* CID font - set glyph specific topdict and private dict */
        resetCIDdicts(_PS_ ttf, cff, aindex);
    }

    err = generateoutline(_PS_ ttf, ptr, size, cffoutl);

    /* restore advance of base char */
    cffoutl->dx = savedx;
    cffoutl->hasadvwidth = savehasadvwidth;

    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ptr);
    return err;
}

/********************************************************************************/
/* since we are not using the PS hints ... there is little to do */
static void stems(int i, CFF_DATA *cffoutl, int which)
{
    int n = (cffoutl->numargs - i) / 2;

    if (which == HSTEM)
        cffoutl->numhstems += n;
    else
        cffoutl->numvstems += n;
}

/********************************************************************************/
/* since we are not using the PS hints ... there is little to do */
static FS_BYTE *do_hintmask(FS_BYTE *ptr, CFF_DATA *cffoutl)
{
    int n = (cffoutl->numhstems + cffoutl->numvstems + 7) / 8;

    return ptr + n;
}

/********************************************************************************/
/* since we are not using the PS hints ... there is little to do */
static FS_BYTE *do_cntrmask(FS_BYTE *ptr, CFF_DATA *cffoutl)
{
    int n = (cffoutl->numhstems + cffoutl->numvstems + 7) / 8;

    return ptr + n;
}

/********************************************************************************/
FS_OUTLINE *loadGlyph(_DS_ TTF *ttf, fsg_SplineKey *key, int index)
{
    FS_BYTE *ptr;
    int size;
    int i, c, w;
    CFF_TAB *cff = ttf->cff;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    CFF_DATA *cffdata = &globalGS->cff_data;
    FS_ULONG err;

    /* first check whether another size has called grow_element() */
    if ( (key->max_points < key->maxp->maxPoints) ||
            (key->max_contours < key->maxp->maxContours))
    {
        /* if so re-calculate offsets, do not increase workspace size */
        /* this was already done for another key                     */
        fsg_WorkSpaceSetOffsets(key);
        fsg_SetUpElement(_PS_ key, GLYPHELEMENT );
        key->max_contours = key->maxp->maxContours;
        key->max_points = key->maxp->maxPoints;
    }

    /* init cffdata */
    cffdata->hasadvwidth = 0;
    cffdata->numargs = 0;
    cffdata->numpoints = 0;
    cffdata->numcontours = 0;
    cffdata->numhstems = 0;
    cffdata->numvstems = 0;
    cffdata->prevx = 0;
    cffdata->prevy = 0;
    cffdata->loopstartx = 0;
    cffdata->loopstarty = 0;
    cffdata->dx = 0;

    /* init elementPtr */
    elementPtr->nc = 0;
    elementPtr->sp[0] = 0;
    elementPtr->ep[0] = 0;

    if (cff->topdict.registry)
    {
        /* CID font - set glyph specific topdict and private dict */
        resetCIDdicts(_PS_ ttf, cff, index);
        if (STATE.error)
            return NULL;
    }
    ptr = (FS_BYTE *)getINDEXptrfromoffset(_PS_ ttf,
                                           ttf->cff_offset + cff->topdict.charstringsoffset,
                                           index, &size);

    /* this should not happen .. but it does ! */
    /* see DFBankBangHK-W5, index=13932        */
    if (ptr == 0 || size <= 0)
        return 0;

    err = generateoutline(_PS_ ttf, ptr, size, cffdata);
    if (err)
    {
        STATE.error = err;
        FSS_free(_PS_ ptr);
        return NULL;
    }
    close_contour(_PS_ cffdata);

    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ptr);

    /* reverse contours in outline -- from PS direction to TT direction */
    for (c = 0; c < elementPtr->nc; c++)
    {
        int s = elementPtr->sp[c];
        int e = elementPtr->ep[c];
        int num = (e - s) / 2;
        FS_LONG temp;        /* can't use an <int> */
        FS_LONG *oox = elementPtr->oox;
        FS_LONG *ooy = elementPtr->ooy;
        FS_BYTE *onCurve = elementPtr->onCurve;

        for (i = 0; i <= num; i++)
        {
            temp = (oox[s + i] + 32) >> 6;
            oox[s + i] = (oox[e - i] + 32) >> 6;
            oox[e - i] = temp;

            temp = (ooy[s + i] + 32) >> 6;
            ooy[s + i] = (ooy[e - i] + 32) >> 6;
            ooy[e - i] = temp;

            temp = onCurve[s + i];
            onCurve[s + i] = onCurve[e - i];
            onCurve[e - i] = (FS_BYTE)temp;
        }
    }

    /* side bearings */
    c = cffdata->numpoints + LEFTSIDEBEARING;
    elementPtr->oox[c] = 0;
    elementPtr->ooy[c] = 0;

    c = cffdata->numpoints + RIGHTSIDEBEARING;
    if (cffdata->hasadvwidth)
        w = ((cffdata->dx + 32) >> 6) + cff->topdict.privatedict.nominalwidthX;
    else
    {
        if (ttf->hmtx_offset)
        {
            fsg_Metrics metrics;
            get_glyph_metrics(_PS_ key, (FS_USHORT)index, &metrics);
            w = metrics.aw;
        }
        else
            w = cff->topdict.privatedict.defaultwidthX;
    }
    elementPtr->oox[c] = w;
    elementPtr->ooy[c] = 0;

    if (STATE.flags & FLAGS_VERTICAL_ON)
    {
        fsg_Metrics metrics;
        F26DOT6 mid;

        mid = (elementPtr->oox[cffdata->numpoints + LEFTSIDEBEARING] +
               elementPtr->oox[cffdata->numpoints + RIGHTSIDEBEARING]) / 2;
        elementPtr->oox[cffdata->numpoints + TOPSIDEBEARING] = mid;
        elementPtr->oox[cffdata->numpoints + BOTTOMSIDEBEARING] = mid;
        elementPtr->oox[cffdata->numpoints + TOPORIGINPOINT] = mid;
        elementPtr->oox[cffdata->numpoints + TOPEDGEPOINT] = mid;

        get_glyph_metrics(_PS_ key, (FS_USHORT)index, &metrics);
        elementPtr->ooy[cffdata->numpoints + TOPSIDEBEARING] = metrics.ah;
        elementPtr->ooy[cffdata->numpoints + BOTTOMSIDEBEARING] = 2 * metrics.ah;
        elementPtr->ooy[cffdata->numpoints + TOPORIGINPOINT] = metrics.ah;
        elementPtr->ooy[cffdata->numpoints + TOPEDGEPOINT] = 0;
    }

    return NULL;
}


/********************************************************************************/
/*                                                                              */
/*  returning everything as FIXED makes more sense                              */
/*  that way the FIXED arguments (b0==255) are not trashed                      */
/********************************************************************************/
static FS_SHORT nextcsopcode( FS_BYTE** pdata, int* numargs, int *args, FS_BYTE *ep)
{
    FS_BYTE *ptr = *pdata;
    FS_BYTE b0, b1, b2, b3, b4;
    FS_LONG count = *numargs;
    FS_LONG val;
    FS_USHORT us;

    for ( ; ; )  /* while (1) */
    {
        if (ptr > ep)
            return -1;

        b0 = *ptr++;
        if (b0 < 28)
        {
            *pdata = ptr;
            *numargs = count;
            return b0;
        }
        else if (b0 == 28)
        {
            b1 = *ptr++;
            b2 = *ptr++;
            us = (b1 << 8) | b2;
            val = (FS_SHORT)us;
            args[count++] = val << 16;
        }
        else if (29 <= b0 && b0 <= 31)
        {
            *pdata = ptr;
            *numargs = count;
            return b0;
        }
        else if (32 <= b0 && b0 <= 246)
        {
            val = b0 - 139;
            args[count++] = val << 16;
        }
        else if (247 <= b0 && b0 <= 250)
        {
            b1 = *ptr++;
            val = (b0 - 247) * 256 + b1 + 108;
            args[count++] = val << 16;
        }
        else if (251 <= b0 && b0 <= 254)
        {
            b1 = *ptr++;
            val = -((b0 - 251) * 256) - b1 - 108;
            args[count++] = val << 16;
        }
        else if (b0 == 255)
        {
            b1 = *ptr++;
            b2 = *ptr++;
            b3 = *ptr++;
            b4 = *ptr++;
            val = (b1 << 24) | (b2 << 16) | (b3 << 8) | b4;
            args[count++] = val;
        }
    }
}

/********************************************************************************/
FS_LONG cff_GridFit(_DS_ fsg_SplineKey *key, FS_BOOLEAN useHints)
{
    TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
    FS_USHORT gIndex = (FS_USHORT)key->glyphIndex;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    int nc, np;

    STATE.level = 0;
    STATE.any_hints = 0;
    STATE.outl_char = 0;    /* it's not an outline char in a stik font */

    /* make sure elementPtr is initialized */
    fsg_SetUpElement(_PS_ key, GLYPHELEMENT );

#ifdef FS_HINTS
    if (key->sfnt != STATE.server->workspace_sfnt)
    {
        /* workspace no longer corresponds to key sfnt, re-run preprogram */
        STATE.server->workspace_sfnt = key->sfnt;

        STATE.error = fsg_RunPreProgram(_PS_ key);
        if (STATE.error)
            return STATE.error;
    }
#endif

#ifdef CFF_DEBUG
    {
        /* zero everything for clarity in debug output */
        int maxpts = ttf->maxp->maxPoints;
        int maxc = ttf->maxp->maxContours;
        SYS_MEMSET(elementPtr->x, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->y, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->ox, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->oy, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->oox, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->ooy, 0, maxpts * 4);
        SYS_MEMSET(elementPtr->f, 0, maxpts);
        SYS_MEMSET(elementPtr->onCurve, 0, maxpts);
        SYS_MEMSET(elementPtr->sp, 0, maxc * 2);
        SYS_MEMSET(elementPtr->ep, 0, maxc * 2);
        elementPtr->nc = 0;
    }
#endif /* CFF_DEBUG */

    loadGlyph(_PS_ ttf, key, gIndex);
    scale_char(key, elementPtr);
    useHints = !(STATE.flags & FLAGS_HINTS_OFF);

    nc = elementPtr->nc;
    np = nc ? 1 + elementPtr->ep[nc - 1] : 0;

#ifdef FS_HINTS
    if ( useHints )
    {
        int ah;

        /* round the LEFTSIDEBEARING */
        elementPtr->x[np + LEFTSIDEBEARING] += 32;
        elementPtr->x[np + LEFTSIDEBEARING] &= ~63;

        if (!(STATE.flags  & FLAGS_GRAYSCALE))
            ah = 0;
        else
        {
            /* should we run the autohinter ? */
            ah = (STATE.flags & FLAGS_FORCE_AUTOHINT);
            ah = ah || (STATE.flags & FLAGS_RTGAH_REF);
            ah = ah || !(STATE.flags & FLAGS_AUTOHINT_OFF);
            /*ah = ah || ((STATE.flags & FLAGS_AUTOHINT_ON) && cffdata->numhstems==0 && cffdata->numvstems==0);*/
            ah = ah && (STATE.cur_sfnt->rtgah_suitable != RTGAH_NOPE);
            ah = ah && (np > 0);
        }

        if (ah)
        {
            FS_BYTE rtptr[3] = {0xb0, 11, 0x7f};
            fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);

            fsg_CopyElementBackwards(elementPtr);
            globalGS->rtgah_data.stateflags = STATE.flags;
            globalGS->sfnt = STATE.cur_sfnt;
            globalGS->rtgah_data.rtgah_suitable = STATE.cur_sfnt->rtgah_suitable;
            globalGS->rtgah_data.fnt_type = STATE.cur_lfnt->fnt_type;
            key->globalGS.rtgah_data.script = which_script(_PS_ gIndex, &(key->globalGS.rtgah_data.unicode));

            fnt_Execute( key->elementInfoRec.interpreterElements, rtptr, rtptr + 3, globalGS);
            STATE.any_hints |= OUTL_FLAGS_RTGAH;
        }

        if (STATE.cur_sfnt->senv && STATE.cur_sfnt->senv->vanilla)
        {
            /* round the RIGHTSIDEBEARING */
            elementPtr->x[np + RIGHTSIDEBEARING] += 32;
            elementPtr->x[np + RIGHTSIDEBEARING] &= ~63;
        }
    }
#endif

    /* apply the base transformation to get rotation/oblique/stretch */
    apply_transMatrix(key, elementPtr->x, elementPtr->y, np + PHANTOMCOUNT, &key->currentTMatrix);

#ifdef CFF_DEBUG
    FS_PRINTF(("cff_Gridfit %d %d\n", gIndex, useHints));
    dump_element("", elementPtr);
#endif

    return STATE.error;
}

/********************************************************************************/
static FS_ULONG generateoutline(_DS_ TTF *ttf, FS_BYTE *ptr, FS_LONG size, CFF_DATA *cffoutl)
{
    FS_BYTE *p;
    FS_BYTE *ep;
    int opcode;
    CFF_TAB *cff = ttf->cff;
    int curx = cffoutl->prevx;
    int cury = cffoutl->prevy;
    int *args = cffoutl->args;
    int numargs;
    FLEXARGS fa;
    int x1 = 0, y1, x2, y2;
    int dx1 = 0, dx2 = 0, dx3 = 0, dx6, dx;
    int dy1 = 0, dy2 = 0, dy3 = 0, dy6, dy;
    int err; /* for lINETO and CURVETO macros */

    /* get max_points and max_contours from key */
    SFNT *sfnt = STATE.cur_sfnt;
    SENV *senv = sfnt->senv;
    fsg_SplineKey *key = (fsg_SplineKey *)senv->ttkey;
    cff->max_points = key->max_points;
    cff->max_contours = key->max_contours;

    if (ptr == 0)
        return SUCCESS;        /* in a strange sort of way */

    p = ptr;
    ep = p + size;

    while (p < ep)
    {
        int i, tmp;

        opcode = nextcsopcode(&p, &cffoutl->numargs, args, ep);
        numargs = cffoutl->numargs;

        if (opcode < 0) return ERR_CFF_BAD_OPCODE;

        i = 0;
        tmp = 0;

        switch ((FS_BYTE)opcode)    /* cast helps compiler to generate better switch table */
        {
        case 1:  /* hstem   */
        case 18: /* hstemhm */
            /* maybe get advance width */
            if (!cffoutl->hasadvwidth && (numargs & 1))
            {
                /* get the difference from the escapement and the nominalWidth */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            /* get the stems */
            stems(i, cffoutl, HSTEM);
            cffoutl->numargs = 0;
            break;

        case 3:
        case 23:
            /* maybe get advance width */
            if (!cffoutl->hasadvwidth && (numargs & 1))
            {
                /* get the difference from the escapement and the nominalWidth */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            /* get the stems */
            stems(i, cffoutl, VSTEM);
            cffoutl->numargs = 0;
            break;

        case 4: /* vmoveto */
            /* maybe get advance width */
            if (!cffoutl->hasadvwidth && (numargs == 2))
            {
                /* get the difference from the escapement and the nominalWidth */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            cury += NEXTARG();
            MOVETO(curx, cury);
            cffoutl->numargs = 0;
            break;

        case 5: /* rlineto */
            /* syntax |- {dxa dya}+ rlineto (5)  */
            while (i < numargs)
            {
                curx += NEXTARG();
                cury += NEXTARG();
                LINETO(curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        case 6: /* hlineto */
            /* syntax 1  |-  dx1 {dya dxb}* hlineto {6} |-                */
            /* syntax 2  |- {dxa  dyb}* hlineto {6} |-                    */
            while (i < numargs)
            {
                if (i & 1)
                    cury += NEXTARG();
                else
                    curx += NEXTARG();
                LINETO(curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        case 7: /* vlineto */
            /* syntax 1 |-  dy1 {dxa dyb}* vlineto (7) |-        */
            /* syntax 2 |- {dya  dxb}* vlineto (7) |-            */
            while (i < numargs)
            {
                if (i & 1)
                    curx += NEXTARG();
                else
                    cury += NEXTARG();
                LINETO(curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        case 8: /* rrcurveto */
            while (i < numargs)
            {
                x1 = curx + NEXTARG();
                y1 = cury + NEXTARG();
                x2 = x1 + NEXTARG();
                y2 = y1 + NEXTARG();
                curx = x2 + NEXTARG();
                cury = y2 + NEXTARG();
                CURVETO(x1, y1, x2, y2, curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        case 10: /* callsubr */
            {
                FS_BYTE *subptr;
                int index = cffoutl->args[cffoutl->numargs - 1] >> 16;
                cffoutl->numargs--;

                cffoutl->prevx = curx;
                cffoutl->prevy = cury;

                subptr = getSubrINDEXptrfromoffset(_PS_ ttf,
                                                   ttf->cff_offset +
                                                   cff->topdict.privatedictoffset +
                                                   cff->topdict.privatedict.subrsoff,
                                                   index, &size);
                if (subptr == NULL)
                    return STATE.error;
                err = generateoutline(_PS_ ttf, subptr, size, cffoutl);
                FSS_free(_PS_ subptr);
                if (err) return err;

                curx = cffoutl->prevx;
                cury = cffoutl->prevy;
            }
            break;

        case 11: /* return */
            /* set these in case we are in a subr */
            cffoutl->prevx = curx;
            cffoutl->prevy = cury;
            return SUCCESS;

        case 14: /* endchar */
            if (numargs == 1 && !cffoutl->hasadvwidth)
            {
                /* space characters ?? */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            if (numargs < 4)
            {
                /* set these in case we are in a subr */
                cffoutl->prevx = curx;
                cffoutl->prevy = cury;
                return SUCCESS;
            }
            else /* numargs >= 4 */
            {
                FS_FIXED asb = 0, adx, ady;
                int bchar, achar;

                if (numargs == 5)
                    asb = NEXTARG();

                adx = NEXTARG();
                ady = NEXTARG();
                bchar = NEXTARGNONCOORD();
                achar = NEXTARGNONCOORD();
                cffoutl->numargs = 0;

                err = processseac(_PS_ ttf, adx, ady, bchar, achar, cffoutl);

                if (numargs == 5)
                {
                    cffoutl->hasadvwidth = 1;
                    cffoutl->dx = asb;
                }
                return err;
            }

        case 12: /*double byte opcodes */
            opcode = *p++;
            switch ((FS_BYTE)opcode)
            {
                /* remember all arguments are fixed point */
            case 3: /* AND */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG((x1 && x2));
                break;
            case 4: /* OR */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG((x1 || x2));
                break;
            case 5: /* NOT */
                x1 = POPCFFARG();
                PUSHCFFARG((x1 == 0));
                break;
            case 9: /*ABS */
                x1 = POPCFFARG();
                if (x1 < 0) x1 = -x1;
                PUSHCFFARG(x1);
                break;
            case 10: /* ADD */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG((x1 + x2));
                break;
            case 11: /* SUB */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG((x1 - x2));
                break;
            case 12: /* DIV */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG( FixDiv(x1, x2) );
                break;
            case 14: /* NEG */
                x1 = POPCFFARG();
                PUSHCFFARG(-x1);
                break;
            case 15: /* EQ */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG((x1 == x2));
                break;
            case 18: /* DROP */
                x2 = POPCFFARG();
                break;
            case 20: /* PUT */
                {
                    int idx = POPCFFARG() >> 16;
                    FS_FIXED v = POPCFFARG();
                    if (idx >= 0 && idx < 32)
                        cffoutl->transient[idx] = v;
                }
                break;
            case 21: /* GET */
                {
                    int idx = POPCFFARG() >> 16;
                    FS_FIXED v = 0;
                    if (idx >= 0 && idx < 32)
                        v = cffoutl->transient[idx];
                    PUSHCFFARG(v);
                }
                break;
            case 22: /* IFELSE */
                y2 = POPCFFARG();
                y1 = POPCFFARG();
                x1 = POPCFFARG();
                x2 = POPCFFARG();
                PUSHCFFARG(((y1 <= y2) ? x1 : x2));
                break;
            case 24: /* MUL */
                x2 = POPCFFARG();
                y2 = POPCFFARG();
                PUSHCFFARG( FixMul(x1, x2) );
                break;
            case 26: /* SQRT */
                x1 = POPCFFARG();
                PUSHCFFARG( FixSqrt(x1) );
                break;
            case 27: /* DUP */
                x1 = POPCFFARG();
                PUSHCFFARG(x1);
                PUSHCFFARG(x1);
                break;
            case 28: /* EXCH */
                x2 = POPCFFARG();
                x1 = POPCFFARG();
                PUSHCFFARG(x2);
                PUSHCFFARG(x1);
                break;
            case 29: /* INDEX */
                x1 = POPCFFARG() >> 16;
                if (x1 >= 0 && x1 < numargs)
                {
                    /* remember top of stack is index==0 */
                    x2 = args[numargs - x1];
                    PUSHCFFARG(x2);
                }
                break;
            case 30: /* ROLL */
                {
                    int temp[48] = {0};
                    int k, J, N;

                    J = POPCFFARG() >> 16;
                    N = POPCFFARG() >> 16;

                    if (N > 0 && N < numargs)
                    {
                        for (i = 0; i < N; i++)
                            temp[i] = POPCFFARG();

                        for (i = 0; i < N; i++)
                        {
                            k = (J + i) % N;
                            PUSHCFFARG(temp[k]);
                        }
                    }
                }
                break;
            case 34: /* hflex */
                /* dx1 dx2 dy2 dx3 dx4 dx5 dx6 hflex */
                fa.x0 = curx;
                fa.y0 = cury;
                fa.x1 = fa.x0 + NEXTARG();
                fa.y1 = cury;
                fa.x2 = fa.x1 + NEXTARG();
                fa.y2 = fa.y1 + NEXTARG();
                fa.x3 = fa.x2 + NEXTARG();
                fa.y3 = fa.y2;
                fa.x4 = fa.x3 + NEXTARG();
                fa.y4 = fa.y2;
                fa.x5 = fa.x4 + NEXTARG();
                fa.y5 = cury;
                fa.x6 = fa.x5 + NEXTARG();
                fa.y6 = cury;
                fa.fd = 50;
                FLEXPROC(&fa);
                curx = fa.x6;
                cury = fa.y6;
                cffoutl->numargs = 0;
                break;
            case 35: /* flex */
                /* dx1 dy1 dx2 dy2 dx3 dy3 dx4 dy4 dx5 dy5 dx6 dy6 fd  flex */
                fa.x0 = curx;
                fa.y0 = cury;
                fa.x1 = fa.x0 + NEXTARG();
                fa.y1 = fa.y0 + NEXTARG();
                fa.x2 = fa.x1 + NEXTARG();
                fa.y2 = fa.y1 + NEXTARG();
                fa.x3 = fa.x2 + NEXTARG();
                fa.y3 = fa.y2 + NEXTARG();
                fa.x4 = fa.x3 + NEXTARG();
                fa.y4 = fa.y3 + NEXTARG();
                fa.x5 = fa.x4 + NEXTARG();
                fa.y5 = fa.y4 + NEXTARG();
                fa.x6 = fa.x5 + NEXTARG();
                fa.y6 = fa.y5 + NEXTARG();
                fa.fd = NEXTARG();
                FLEXPROC(&fa);
                curx = fa.x6;
                cury = fa.y6;
                cffoutl->numargs = 0;
                break;
            case 36: /* hflex1 */
                /* dx1 dy1 dx2 dy2 dx3 dx4 dx5 dy5 dx6 hflex1 */
                fa.x0 = curx;
                fa.y0 = cury;
                fa.x1 = fa.x0 + NEXTARG();
                fa.y1 = fa.y0 + NEXTARG();
                fa.x2 = fa.x1 + NEXTARG();
                fa.y2 = fa.y1 + NEXTARG();
                fa.x3 = fa.x2 + NEXTARG();
                fa.y3 = fa.y2;
                fa.x4 = fa.x3 + NEXTARG();
                fa.y4 = fa.y2;
                fa.x5 = fa.x4 + NEXTARG();
                fa.y5 = fa.y4 + NEXTARG();
                fa.x6 = fa.x5 + NEXTARG();
                fa.y6 = cury;
                fa.fd = 50;
                FLEXPROC(&fa);
                curx = fa.x6;
                cury = fa.y6;
                cffoutl->numargs = 0;
                break;
            case 37: /* flex1 */
                /* dx1 dy1 dx2 dy2 dx3 dy3 dx4 dy4 dx5 dy5 d6 flex1 */
                fa.x0 = curx;
                fa.y0 = cury;
                fa.x1 = fa.x0 + NEXTARG();
                fa.y1 = fa.y0 + NEXTARG();
                fa.x2 = fa.x1 + NEXTARG();
                fa.y2 = fa.y1 + NEXTARG();
                fa.x3 = fa.x2 + NEXTARG();
                fa.y3 = fa.y2 + NEXTARG();
                fa.x4 = fa.x3 + NEXTARG();
                fa.y4 = fa.y3 + NEXTARG();
                fa.x5 = fa.x4 + NEXTARG();
                fa.y5 = fa.y4 + NEXTARG();
                dx6 = dy6 = 0;
                dx = fa.x5 - fa.x0;
                dy = fa.y5 - fa.y0;
                if (abs(dx) > abs(dy))
                    dx6 = NEXTARG();
                else
                    dy6 = NEXTARG();
                fa.x6 = fa.x5 + dx6;
                fa.y6 = fa.y5 + dy6;
                fa.fd = 50;
                FLEXPROC(&fa);
                curx = fa.x6;
                cury = fa.y6;
                cffoutl->numargs = 0;
                break;
            default:
                /* FS_PRINTF(("generateoutline : bad opcode 12 %d\n",opcode)); */
                break;
            }  /* end of case 12's switch */
            break; /* end of case 12: */

        case 19: /* hintmask */
            /* if there are any arguments they are VSTEMS */
            if (numargs)
            {
                stems(0, cffoutl, VSTEM);
                cffoutl->numargs = 0;
            }

            /* then the hintmask proper */
            p = do_hintmask(p, cffoutl);
            break;

        case 20: /* cntrmask */
            /* if there are any arguments they are VSTEMS */
            if (numargs)
            {
                stems(0, cffoutl, VSTEM);
                cffoutl->numargs = 0;
            }

            /* then the hintmask proper */
            p = do_cntrmask(p, cffoutl);
            cffoutl->numargs = 0;
            break;

        case 21: /* rmoveto */
            /* syntax |- dx1 dy1 rmoveto (21) |- */
            /* maybe get advance width */
            if (!cffoutl->hasadvwidth && (numargs == 3))
            {
                /* get the difference from the escapement and the nominalWidth */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            curx += NEXTARG();
            cury += NEXTARG();
            MOVETO(curx, cury);
            cffoutl->numargs = 0;
            break;

        case 22: /* hmoveto */
            /* syntax |- dx1 hmoveto |- */
            /* maybe get advance width  */
            if (!cffoutl->hasadvwidth && (numargs == 2))
            {
                /* get the difference from the escapement and the nominalWidth */
                cffoutl->dx = NEXTARG();
                cffoutl->hasadvwidth = 1;
            }
            curx += NEXTARG();
            MOVETO(curx, cury);
            cffoutl->numargs = 0;
            break;

        case 24: /* rcurveline */
            /* syntax |- {dxa dya dxb dyb dxc dyc}+ dxd dyd rcurveline (24) |- */
            tmp = numargs - 2;
            while (i < tmp)
            {
                x1 = curx + NEXTARG();
                y1 = cury + NEXTARG();
                x2 = x1 + NEXTARG();
                y2 = y1 + NEXTARG();
                curx = x2 + NEXTARG();
                cury = y2 + NEXTARG();
                CURVETO(x1, y1, x2, y2, curx, cury);
            }
            curx += NEXTARG();
            cury += NEXTARG();
            LINETO(curx, cury);
            cffoutl->numargs = 0;
            break;

        case 25: /* rlinecurve */
            /* syntax |- {dxa dya}+ dxb dyb dxc dyc dxd dyd rlinecurve (25) |- */
            tmp = numargs - 6;
            while (i < tmp)
            {
                curx += NEXTARG();
                cury += NEXTARG();
                LINETO(curx, cury);
            }
            x1 = curx + NEXTARG();
            y1 = cury + NEXTARG();
            x2 = x1 + NEXTARG();
            y2 = y1 + NEXTARG();
            curx = x2 + NEXTARG();
            cury = y2 + NEXTARG();
            CURVETO(x1, y1, x2, y2, curx, cury);
            cffoutl->numargs = 0;
            break;

        case 26: /* vvcurveto */
            /* syntax |- dx1? {dy1 dx2 dy2 dy3}+ vvcurveto (26) |-  */
            dx1 = dx3 = 0;
            if (numargs & 1)
                dx1 = NEXTARG();
            while (i < numargs)
            {
                dy1 = NEXTARG();
                dx2 = NEXTARG();
                dy2 = NEXTARG();
                dy3 = NEXTARG();
                x1 = curx + dx1;
                y1 = cury + dy1;
                x2 = x1 + dx2;
                y2 = y1 + dy2;
                curx = x2 + dx3;
                cury = y2 + dy3;
                CURVETO(x1, y1, x2, y2, curx, cury);
                dx1 = 0;
            }
            cffoutl->numargs = 0;
            break;

        case 27: /* hhcurveto */
            /* syntax  |- dy1? {dx1 dx2 dy2 dx3}+ hhcurveto |- */
            dy1 = dy3 = 0;
            if (numargs & 1)
                dy1 = NEXTARG();
            while (i < numargs)
            {
                dx1 = NEXTARG();
                dx2 = NEXTARG();
                dy2 = NEXTARG();
                dx3 = NEXTARG();
                x1 = curx + dx1;
                y1 = cury + dy1;
                x2 = x1 + dx2;
                y2 = y1 + dy2;
                curx = x2 + dx3;
                cury = y2 + dy3;
                CURVETO(x1, y1, x2, y2, curx, cury);
                dy1 = 0;
            }
            cffoutl->numargs = 0;
            break;

        case 28: /* short int */
            tmp = *ptr++;
            tmp = (tmp << 8) | *ptr++;
            PUSHCFFARG((FS_SHORT)tmp);
            break;

        case 29: /* call gsbr */
            {
                FS_BYTE *subptr;
                int index;
                FS_LONG asize = 0;

                index = cffoutl->args[--cffoutl->numargs] >> 16;

                cffoutl->prevx = curx;
                cffoutl->prevy = cury;

                subptr = getSubrINDEXptr(_PS_ cff->globalsubrINDEXptr, index, &asize);
                if (subptr == NULL)
                    return STATE.error;

                err = generateoutline(_PS_ ttf, subptr, asize, cffoutl);
                if (err) return err;

                curx = cffoutl->prevx;
                cury = cffoutl->prevy;
            }
            break;

        case 30: /* vhcurveto */
            /* syntax1 |-  dy1 dx2 dy3 dx3 {dxa dxb dyb dyc  dyd dxe dye dxf}* dyf? vhcurveto|-  */
            /* syntax2 |- {dya dxb dyb dxc  dxd dxe dye dyf}+ dxf? vhcurveto |-   */
            while (i < numargs)
            {
                tmp = (i + 4) % 8;
                if (tmp == 4)
                {
                    dy1 = NEXTARG();
                    dx2 = NEXTARG();
                    dy2 = NEXTARG();
                    dx3 = NEXTARG();
                    dx1 = dy3 = 0;
                }
                else if (tmp == 0)
                {
                    dx1 = NEXTARG();
                    dx2 = NEXTARG();
                    dy2 = NEXTARG();
                    dy3 = NEXTARG();
                    dy1 = dx3 = 0;
                }
                /* optional last args */
                if ((numargs - i) == 1)
                {
                    if ((numargs % 8) == 1)
                        dx3 = NEXTARG();
                    else if ((numargs % 4) == 1)
                        dy3 = NEXTARG();
                }

                x1 = curx + dx1;
                y1 = cury + dy1;
                x2 = x1 + dx2;
                y2 = y1 + dy2;
                curx = x2 + dx3;
                cury = y2 + dy3;
                CURVETO(x1, y1, x2, y2, curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        case 31: /* hvcurveto */
            /* syntax1 |-  dx1 dx2 dy2 dy3 {dya dxb dyb dxc   dxd dxe dye dyf}* dxf? hvcurveto |- */
            /* syntax2 |- {dxa dxb dyb dyc  dyd dxe dye dxf}+ dyf? hvcurveto |-    */
            while (i < numargs)
            {
                tmp = (i + 4) % 8;
                if (tmp == 4)
                {
                    dx1 = NEXTARG();
                    dx2 = NEXTARG();
                    dy2 = NEXTARG();
                    dy3 = NEXTARG();
                    dy1 = dx3 = 0;
                }
                else if (tmp == 0)
                {
                    dy1 = NEXTARG();
                    dx2 = NEXTARG();
                    dy2 = NEXTARG();
                    dx3 = NEXTARG();
                    dx1 = dy3 = 0;
                }
                if ((numargs - i) == 1)
                {
                    if ((numargs % 8) == 1)
                        dy3 = NEXTARG();
                    else if ((numargs % 4) == 1)
                        dx3 = NEXTARG();
                }
                x1 = curx + dx1;
                y1 = cury + dy1;
                x2 = x1 + dx2;
                y2 = y1 + dy2;
                curx = x2 + dx3;
                cury = y2 + dy3;
                CURVETO(x1, y1, x2, y2, curx, cury);
            }
            cffoutl->numargs = 0;
            break;

        default:
            /* FS_PRINTF(("generateoutline : bad opcode %d\n",opcode)); */
            break;
        }
    }

    return SUCCESS;
}

#endif /* ifdef-FS_RENDER */

/********************************************************************************/

FS_LONG int_arg(ARGS *args, int n)
{
    FS_LONG v = args->val[n];
    FS_BYTE t = args->type[n];
    if (t == ARG_TYPE_FIXED)
        v >>= 16;
    else if (t == ARG_TYPE_FRACT)
        v >>= 30;
    return v;
}

/* 2.30 */
FRACT fract_arg(ARGS *args, int n)
{
    FS_LONG v = args->val[n];
    FS_BYTE t = args->type[n];
    if (t == ARG_TYPE_INT)
        v <<= 30;
    else if (t == ARG_TYPE_FIXED)
        v <<= 14;
    return v;
}

/********************************************************************************/

#define IS_DIGIT(x) (x>='0' && x<='9')

/* get real from stream, return as int, fixed, or FRACT as needed */
/* ugly little routine when we can't use floating point... */
static FS_BYTE *get_real(FS_BYTE *p, ARGS *args, int argcount)
{
    int i, k;
    FS_BYTE v, w, string[128];
    FS_CONST char *pattern = "0123456789.EE?-";
    FS_BYTE digits[32];
    int dot = 0;
    int neg = 0;
    int exp_neg = 0;
    int exp = 0;
    int count = 0;
    FS_LONG integer;
    FS_LONG fraction;
    FS_LONG val;

    /* get the real as a string */
    for (;;)
    {
        v = *p++;
        w = 0;
        for (k = 0; k < 2; k++)
        {
            if (k == 0) w = v >> 4;
            else      w = v & 0xF;

            string[count++] = pattern[w];
            if (w == 0xc) string[count++] = '-';
            if (w == 0xF) break;
        }
        if (w == 0xF) break;
    }

    /* get mantissa and exponent */
    i = 0;
    if (string[i] == '-')
    {
        neg = 1;
        i++;
    }
    while (IS_DIGIT(string[i]))
        digits[k++] = string[i++];
    if (string[i] == '.')
    {
        dot = k;
        i++;
    }
    while (IS_DIGIT(string[i]))
        digits[k++] = string[i++];

    if (string[i] == 'E')
    {
        i++;
        if (string[i] == '-')
        {
            exp_neg = 1;
            i++;
        }
        while (IS_DIGIT(string[i]))
            exp = (10 * exp) + (string[i] - '0');
        if (exp_neg)
            exp = -exp;
    }
    digits[k] = 0;

    /* get the integer portion */
    integer = 0;
    dot += exp;
    i = 0;
    while (i < dot)
        integer = (integer * 10) + (digits[i++] - '0');

    /* get the fraction portion */
    fraction = 0;
    exp = 1;    /* the denumerator of the fraction */
    while (dot < 0)
    {
        dot++;
        exp *= 10;
    }
    while (dot < k)
    {
        fraction = (fraction * 10) + (digits[dot++] - '0');
        exp *= 10;
    }

    if (integer >= 32768)
    {
        /* is INT result */
        val = neg ? -integer : integer;
        args->val[argcount] = val;
        args->type[argcount] = ARG_TYPE_INT;
    }
    else if (integer >= 2)
    {
        /* is FIXED result */
        val = integer << 16;
        val += LongMulDiv(fraction, (1 << 16), exp);
        if (neg) val = -val;
        args->val[argcount] = val;
        args->type[argcount] = ARG_TYPE_FIXED;
    }
    else
    {
        /* is FRACT result */
        val = integer << 30;
        val += LongMulDiv(fraction, (1 << 30), exp);
        if (neg) val = -val;
        args->val[argcount] = val;
        args->type[argcount] = ARG_TYPE_FRACT;
    }

    return p;
}

static void loadCharSet(_DS_ TTF *ttf, CFF_TAB *cff)
{
    CFFTOPDICT *topdict = &cff->topdict;
    FS_USHORT numglyphs;
    FS_ULONG offset = ttf->cff_offset + cff->topdict.charsetoffset;

    numglyphs = topdict->numglyphs;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "loadCharSet";
#endif
    topdict->charset = (FS_USHORT *)FSS_calloc(_PS_ sizeof(FS_USHORT) * numglyphs);
    if (topdict->charset == NULL)
        return; /* with state error */

    if (cff->topdict.charsetoffset < 3) /* use predefined charset */
    {
        FS_CONST FS_USHORT *ranges;
        FS_USHORT first, last;
        int i, k;
        switch (cff->topdict.charsetoffset)
        {
        case 0:
            ranges = ISOAdobeSID;
            break;
        case 1:
            ranges = ExpertSID;
            break;
        case 2:
            ranges = ExpertSubsetSID;
            break;
        default:
            ranges = ISOAdobeSID;
            break;
        }
        i = 1; /* glyph index */
        k = 0; /* range index */
        first = ranges[k];
        last  = ranges[k + 1];
        while (first != 0 && last != 0)
        {
            int j;
            for (j = first; j <= last && i < numglyphs; j++)
            {
                topdict->charset[i++] = (FS_USHORT)j;
            }
            k += 2;
            first = ranges[k];
            last  = ranges[k + 1];
        }
    }
    else /* read charset data */
    {
        FS_BYTE format;

        ttf_read_buf(_PS_ ttf, offset, 1, &format);
        offset += 1;

        if (format == 0)
        {
            FS_USHORT i;
            FS_USHORT *sid_array, *sid;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "loadCharSet";
#endif
            sid_array = (FS_USHORT *)ttf_read(_PS_ ttf, offset, sizeof(FS_USHORT) * (numglyphs - 1));
            if (sid_array == 0)
                return;

            i = 1; /* missing glyph is omitted */
            sid = sid_array;
            while (i < numglyphs)
            {
                topdict->charset[i++] = *sid++;
            }
            FSS_free(_PS_ sid_array);
        }
        else if (format == 1 || format == 2)
        {
            FS_USHORT i, first, nLeft;
            FS_BYTE bLeft;

            i = 1; /* missing glyph is omitted */
            while (i < numglyphs)
            {
                FS_USHORT j;

                ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&first);
                first = SWAPW(first);
                offset += 2;
                if (format == 1)
                {
                    ttf_read_buf(_PS_ ttf, offset, 1, &bLeft);
                    nLeft = (FS_USHORT)bLeft;
                    offset += 1;
                }
                else /* format 2 */
                {
                    ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&nLeft);
                    nLeft = SWAPW(nLeft);   /* nLeft stored in 2 bytes */
                    offset += 2;
                }

                /* check whether charset is simple                        */
                /* a simple charset is one where the sid equals the index */
                /* no need to store simple charsets                       */
                if ( (first == 1) && (nLeft == numglyphs - 2) )
                {
                    FSS_free(_PS_ topdict->charset);
                    topdict->charset = 0;
                    return;
                }

                for (j = first; j <= first + nLeft && i < numglyphs; j++)
                {
                    topdict->charset[i++] = j;
                }
            }
        } /* else format 1 or 2 */
    } /* else read charset data */
}
/********************************************************************************/
static int getINDEXsizefromoffset(_DS_ TTF *ttf, int off, FS_USHORT *count)
{
    FS_BYTE entrysize;
    int i, size, offset;
    unsigned char tmp[4];

    ttf_read_buf(_PS_ ttf, off, 3, tmp);
    *count = (tmp[0] << 8 ) | tmp[1];
    if (*count == 0)
        return 0; /* empty index */

    entrysize = tmp[2];
    if (entrysize > 4)
    {
        STATE.error = ERR_CFF_BAD_OFFSET_SIZE;
        return 0;
    }

    ttf_read_buf(_PS_ ttf, off + 3 + (*count)*entrysize, 4, tmp);

    for (offset = 0, i = 0; i < entrysize; i++)
        offset = (offset << 8) + tmp[i];

    size = 3 + (*count + 1) * entrysize + offset - 1;

    return size;
}

/********************************************************************************/
static void *getINDEXptr(FS_BYTE *ptr, int index, FS_ULONG *size)
{
    FS_USHORT count;
    int entrysize;
    int i, offset, nextoffset;
    FS_BYTE *p = ptr;

    count = (p[0] << 8) | p[1];
    entrysize = p[2];
    p += 3;

    if (count < index)
        return NULL;

    for (offset = 0, nextoffset = 0, i = 0; i < entrysize; i++)
    {
        offset = (offset << 8) + *(p + index * entrysize + i);
        nextoffset = (nextoffset << 8) + *(p + (index + 1) * entrysize + i);
    }

    *size = nextoffset - offset;
    return (p + (count + 1) * entrysize - 1 + offset);
}

/********************************************************************************/
static void initTopDict(CFFTOPDICT *topdict)
{
    SYS_MEMSET(topdict, 0, sizeof(CFFTOPDICT));
    topdict->fontmatrix[0] = 1073742; /* .001 as 2.30 FRACT */
    topdict->fontmatrix[3] = 1073742; /* .001 as 2.30 FRACT */
    topdict->underlinethickness = -100;
    topdict->underlineposition = 50;
    topdict->charstringtype = 2;
}

/********************************************************************************/
/* since we are not using the PS hints ... this is easy */
static void loadPrivateDict(FS_BYTE *ptr, FS_LONG size, PRIVATEDICT *privatedict)
{
    ARGS args;
    FS_LONG argcount = 0;
    FS_LONG opcode;
    FS_BYTE *p = ptr;
    FS_BYTE *ep = ptr + size;

    privatedict->subrsoff = 0;
    privatedict->defaultwidthX = 0;
    privatedict->nominalwidthX = 0;

    while (p < ep)
    {
        opcode = nextopcode(&p, &argcount, &args, ep);

        if (opcode == 19)
            privatedict->subrsoff = int_arg(&args, 0);

        else if (opcode == 20)
            privatedict->defaultwidthX = int_arg(&args, 0);

        else if (opcode == 21)
            privatedict->nominalwidthX = int_arg(&args, 0);

        argcount = 0;
    }
}

/********************************************************************************/
/* since we are not using the PS hints this is easy */
static void loadTopDict(TTF *ttf, FS_LONG size, FS_BYTE *ptr, CFFTOPDICT *topdict)
{
    ARGS args;
    FS_LONG argcount = 0;
    int opcode;

    FS_BYTE *p;
    FS_BYTE *ep;

    if (ptr == 0)
        return;

    p = ptr;
    ep = ptr + size;

    while (p < ep)
    {
        opcode = nextopcode(&p, &argcount, &args, ep);
        if (opcode == -1)
            return;

        if (opcode == 5)
        {
            topdict->fontbbox[0] = int_arg(&args, 0);
            topdict->fontbbox[1] = int_arg(&args, 1);
            topdict->fontbbox[2] = int_arg(&args, 2);
            topdict->fontbbox[3] = int_arg(&args, 3);
        }
        else if (opcode == 12)
        {
            opcode = *p++;

            if (opcode == 6)
                topdict->charstringtype = int_arg(&args, 0);
            else if (opcode == 7)
            {
                topdict->fontmatrix[0] = fract_arg(&args, 0);
                topdict->fontmatrix[1] = fract_arg(&args, 1);
                topdict->fontmatrix[2] = fract_arg(&args, 2);
                topdict->fontmatrix[3] = fract_arg(&args, 3);
                topdict->fontmatrix[4] = fract_arg(&args, 4);
                topdict->fontmatrix[5] = fract_arg(&args, 5);
            }
            else if (opcode == 30)
            {
                /* ROS */
                topdict->registry = (FS_SHORT)int_arg(&args, 0);
                topdict->FDselect = -1;
                /* topdict->ordering = (FS_SHORT)int_arg(&args,1); */
                /* topdict->supplement = int_arg(&args,2); */
            }
            else if (opcode == 34)
            {
                topdict->CIDcount = int_arg(&args, 0);
            }
            else if (opcode == 36)
            {
                topdict->FDarrayoffset = int_arg(&args, 0);
                topdict->FDarrayoffset += ttf->cff_offset;
            }
            else if (opcode == 37)
            {
                topdict->FDselectoffset = int_arg(&args, 0);
                topdict->FDselectoffset += ttf->cff_offset;
            }
        }
        else if (opcode == 15)
            topdict->charsetoffset = int_arg(&args, 0);
        else if (opcode == 16)
            topdict->encodingoffset = int_arg(&args, 0);
        else if (opcode == 17)
            topdict->charstringsoffset = int_arg(&args, 0);
        else if (opcode == 18)
        {
            topdict->privatedictsize = int_arg(&args, 0);
            topdict->privatedictoffset = int_arg(&args, 1);
        }
        argcount = 0;
    }

    /* now divide all elements of the fontmatrix by fontmatrix[0] */
    topdict->fontmatrix[1] = VarDiv(topdict->fontmatrix[1], topdict->fontmatrix[0], 30);
    topdict->fontmatrix[2] = VarDiv(topdict->fontmatrix[2], topdict->fontmatrix[0], 30);
    topdict->fontmatrix[3] = VarDiv(topdict->fontmatrix[3], topdict->fontmatrix[0], 30);
    topdict->fontmatrix[4] = VarDiv(topdict->fontmatrix[4], topdict->fontmatrix[0], 30);
    topdict->fontmatrix[5] = VarDiv(topdict->fontmatrix[5], topdict->fontmatrix[0], 30);
    topdict->fontmatrix[0] = 1 << 30;  /* 1.00 in FRACT */
}

/********************************************************************************/
static FS_SHORT nextopcode( FS_BYTE** pdata, FS_LONG* numargs, ARGS *args, FS_BYTE *ep)
{
    FS_BYTE *ptr = *pdata;
    FS_BYTE b0, b1, b2, b3, b4;
    FS_LONG count = *numargs;
    FS_LONG val;
    FS_USHORT us;
    FS_SHORT s;

    for ( ; ; )  /* while (1) */
    {
        if (ptr > ep)
            return -1;

        b0 = *ptr++;
        if (b0 < 28)
        {
            *pdata = ptr;
            *numargs = count;
            return b0;
        }
        else if (b0 == 28)
        {
            b1 = *ptr++;
            b2 = *ptr++;
            us = (b1 << 8) | b2;
            s = (FS_SHORT) us;
            args->val[count] = s;
            args->type[count++] = ARG_TYPE_INT;
        }
        else if (b0 == 29)
        {
            b1 = *ptr++;
            b2 = *ptr++;
            b3 = *ptr++;
            b4 = *ptr++;
            val = (b1 << 24) | (b2 << 16) | (b3 << 8) | b4;
            args->val[count] = val;
            args->type[count++] = ARG_TYPE_INT;
        }
        else if (b0 == 30)
        {
            /* arg is one of 3 arg types */
            ptr = get_real(ptr, args, count++);
        }
        else if (32 <= b0 && b0 <= 246)
        {
            val = b0 - 139;
            args->val[count] = val;
            args->type[count++] = ARG_TYPE_INT;
        }
        else if (247 <= b0 && b0 <= 250)
        {
            b1 = *ptr++;
            val = (b0 - 247) * 256 + b1 + 108;
            args->val[count] = val;
            args->type[count++] = ARG_TYPE_INT;
        }
        else if (251 <= b0 && b0 <= 254)
        {
            b1 = *ptr++;
            val = -(b0 - 251) * 256 - b1 - 108;
            args->val[count] = val;
            args->type[count++] = ARG_TYPE_INT;
        }
        else
        {
            /* FS_PRINTF(("nextopcode b0 = %d\n",b0));*/
        }
    } /* while (1) */
}

/********************************************************************************/
CFF_TAB *load_cff(_DS_  TTF *ttf)
{
    CFF_TAB *cff;
    FS_BYTE tmp[4];
    FS_ULONG offset, size;
    FS_USHORT count;
    int num;
    FS_BYTE *topdictptr;
    FS_ULONG headersize;

    if (!get_ttf_table_info(_PS_ ttf, TAG_cff, &offset, &size))
        return 0;
    ttf->cff_offset = offset;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "load_cff";
#endif
    cff = FSS_malloc(_PS_ sizeof(CFF_TAB));
    if (cff == 0 || STATE.error)
        return 0;
    ttf_read_buf(_PS_ ttf, offset, 4, tmp);

    headersize = tmp[2];

    /* Read Name INDEX */
    cff->nameINDEXoffset = offset + headersize;
    size = getINDEXsizefromoffset(_PS_ ttf, cff->nameINDEXoffset, &count);
    if (STATE.error)
    {
        FSS_free(_PS_ cff);
        return 0;
    }

    /* can be only one entry in the name index */
    ttf_read_buf(_PS_ ttf, cff->nameINDEXoffset, 2, tmp);
    num = (tmp[0] << 8) | tmp[1];
    if (num != 1)
    {
        STATE.error = ERR_CFF_MULTIPLE_FONTS;
        FSS_free(_PS_ cff);
        return 0;
    }

    /* read Top DICT INDEX */
    cff->topdictINDEXoffset = cff->nameINDEXoffset + size;
    size = getINDEXsizefromoffset(_PS_ ttf, cff->topdictINDEXoffset, &count);
    if (STATE.error)
    {
        FSS_free(_PS_ cff);
        return 0;
    }

    cff->topdictINDEXptr = ttf_read(_PS_ ttf, cff->topdictINDEXoffset, size);
    if (cff->topdictINDEXptr == 0 || STATE.error)
    {
        FSS_free(_PS_ cff);
        return 0;
    }
    cff->stringINDEXoffset = cff->topdictINDEXoffset + size;
    size = getINDEXsizefromoffset(_PS_ ttf, cff->stringINDEXoffset, &count);
    if (STATE.error)
    {
        FSS_free(_PS_ cff);
        return 0;
    }

    /* read Global subr INDEX */
    cff->globalsubrINDEXoffset = cff->stringINDEXoffset + size;
    size = getINDEXsizefromoffset(_PS_ ttf, cff->globalsubrINDEXoffset, &count);
    if (STATE.error)
    {
        FSS_free(_PS_ cff);
        return 0;
    }

    cff->globalsubrINDEXptr = ttf_read(_PS_ ttf, cff->globalsubrINDEXoffset, size);

    topdictptr = getINDEXptr(cff->topdictINDEXptr, 0, &size);
    initTopDict(&cff->topdict);
    loadTopDict(ttf, size, topdictptr, &cff->topdict);

    if (cff->topdict.charstringtype != 2)
    {
        STATE.error = ERR_CFF_CHARSTRINGTYPE;
        unload_cff(_PS_ cff);
        return 0;
    }

    /* the charstring index is located by the cff->topdict.charstringoffset */
    cff->charstringINDEXoffset = ttf->cff_offset + cff->topdict.charstringsoffset;
    size = getINDEXsizefromoffset(_PS_ ttf, cff->charstringINDEXoffset, &count);
    if (STATE.error)
    {
        unload_cff(_PS_ cff);
        return 0;
    }

    /* number of glyphs in font is the count of the charstring index */
    cff->topdict.numglyphs = count;

    if (cff->topdict.charsetoffset)
        loadCharSet(_PS_ ttf, cff);

    /* if there is a private dict, load it */
    if (cff->topdict.privatedictsize)
    {
        FS_BYTE *privatedictptr;

        offset = ttf->cff_offset + cff->topdict.privatedictoffset;
        size = cff->topdict.privatedictsize;
        privatedictptr = ttf_read(_PS_ ttf, offset, size);
        loadPrivateDict(privatedictptr, size, &cff->topdict.privatedict);
        FSS_free(_PS_ privatedictptr);
    }

    return cff;
}

/********************************************************************************/
/* free everything that was obtained by FSS_malloc or ttf_read() in load_cff()  */
void unload_cff(_DS_ CFF_TAB *cff)
{
#ifdef FS_MEM_DBG
    STATE.memdbgid = "unload_cff";
#endif
    if (cff)
    {
        if (cff->topdictINDEXptr)
            FSS_free(_PS_ cff->topdictINDEXptr);

        if (cff->globalsubrINDEXptr)
            FSS_free(_PS_ cff->globalsubrINDEXptr);

        if (cff->topdict.charset)
            FSS_free(_PS_ cff->topdict.charset);

        FSS_free(_PS_ cff);
    }
}
#endif /* ifdef-FS_CFFR: conditionally compile entire module */
